<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['client_user_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$clientId = $_SESSION['client_id'];

// Filters
$filterStatus = $_GET['status'] ?? '';
$filterDate = $_GET['date'] ?? '';

$query = "
    SELECT p.*, 
           m.meter_id,
           cu.customer_id, CONCAT(cu.first_name, ' ', cu.last_name) as customer_name,
           t.token_value
    FROM payments p
    LEFT JOIN meters m ON p.meter_id = m.id
    LEFT JOIN customers cu ON p.customer_id = cu.id
    LEFT JOIN tokens t ON p.id = t.payment_id
    WHERE p.client_id = ?
";

$params = [$clientId];
if ($filterStatus) {
    $query .= " AND p.status = ?";
    $params[] = $filterStatus;
}
if ($filterDate) {
    $query .= " AND DATE(p.tx_date) = ?";
    $params[] = $filterDate;
}

$query .= " ORDER BY p.tx_date DESC LIMIT 100";

$stmt = $db->prepare($query);
$stmt->execute($params);
$payments = $stmt->fetchAll();

// Statistics
$stats = [
    'total' => $db->prepare("SELECT COUNT(*) as c FROM payments WHERE client_id = ?"),
    'today' => $db->prepare("SELECT COUNT(*) as c FROM payments WHERE client_id = ? AND DATE(tx_date)=CURDATE()"),
    'processed' => $db->prepare("SELECT COUNT(*) as c FROM payments WHERE client_id = ? AND status='PROCESSED'"),
    'pending' => $db->prepare("SELECT COUNT(*) as c FROM payments WHERE client_id = ? AND status='PENDING'"),
    'today_revenue' => $db->prepare("SELECT COALESCE(SUM(amount),0) as c FROM payments WHERE client_id = ? AND DATE(tx_date)=CURDATE() AND status='PROCESSED'")
];

foreach ($stats as $key => $stmt) {
    $stmt->execute([$clientId]);
    $stats[$key] = $stmt->fetch()['c'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment History</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Payment History</h1>
                <div class="flex items-center gap-4">
                    <span><?= htmlspecialchars($_SESSION['client_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <!-- Statistics -->
                <div class="grid grid-cols-3 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Total Payments</h3>
                        <div class="value"><?= $stats['total'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Today</h3>
                        <div class="value"><?= $stats['today'] ?></div>
                    </div>
                    <div class="stat-card">
                        <h3>Today Revenue</h3>
                        <div class="value">KES <?= number_format($stats['today_revenue'], 2) ?></div>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Processed</h3>
                        <div class="value"><?= $stats['processed'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Pending</h3>
                        <div class="value"><?= $stats['pending'] ?></div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="bg-white p-4 rounded shadow mb-4">
                    <form method="GET" class="grid grid-cols-3 gap-4">
                        <div class="form-group">
                            <label>Status</label>
                            <select name="status">
                                <option value="">All Status</option>
                                <option value="PENDING" <?= $filterStatus == 'PENDING' ? 'selected' : '' ?>>Pending</option>
                                <option value="VERIFIED" <?= $filterStatus == 'VERIFIED' ? 'selected' : '' ?>>Verified</option>
                                <option value="PROCESSED" <?= $filterStatus == 'PROCESSED' ? 'selected' : '' ?>>Processed</option>
                                <option value="FAILED" <?= $filterStatus == 'FAILED' ? 'selected' : '' ?>>Failed</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Date</label>
                            <input type="date" name="date" value="<?= htmlspecialchars($filterDate) ?>">
                        </div>
                        <div class="form-group" style="display:flex;align-items:flex-end;gap:0.5rem">
                            <button type="submit" class="btn btn-primary">Filter</button>
                            <a href="payments.php" class="btn">Reset</a>
                        </div>
                    </form>
                </div>
                
                <!-- Payments Table -->
                <div class="table-container">
                    <div class="table-header">
                        <h2>Payment History (Last 100)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>M-Pesa Ref</th>
                                <th>Meter</th>
                                <th>Customer</th>
                                <th>Phone</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Token</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($payments as $p): ?>
                                <tr>
                                    <td><?= date('Y-m-d H:i', strtotime($p['tx_date'])) ?></td>
                                    <td><code style="font-size:0.7rem"><?= htmlspecialchars($p['mpesa_tx_id']) ?></code></td>
                                    <td><?= htmlspecialchars($p['meter_id'] ?: 'N/A') ?></td>
                                    <td><?= htmlspecialchars($p['customer_name'] ?: 'N/A') ?></td>
                                    <td><?= htmlspecialchars($p['phone']) ?></td>
                                    <td><strong>KES <?= number_format($p['amount'], 2) ?></strong></td>
                                    <td>
                                        <?php
                                        $colors = [
                                            'PROCESSED' => 'badge-success',
                                            'PENDING' => 'badge-warning',
                                            'VERIFIED' => 'badge-info',
                                            'FAILED' => 'badge-danger'
                                        ];
                                        ?>
                                        <span class="badge <?= $colors[$p['status']] ?? 'badge-info' ?>">
                                            <?= $p['status'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($p['token_value']): ?>
                                            <code style="font-size:0.7rem"><?= htmlspecialchars(substr($p['token_value'], 0, 15)) ?>...</code>
                                        <?php else: ?>
                                            <span class="badge badge-warning">Pending</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button onclick="viewPayment(<?= htmlspecialchars(json_encode($p)) ?>)" class="btn btn-sm btn-primary">
                                            <i class="fas fa-eye"></i> View
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- View Payment Modal -->
    <div id="viewModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Payment Details</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <div class="modal-body" id="paymentDetails">
                <!-- Populated by JS -->
            </div>
            <div class="modal-footer">
                <button onclick="closeModal()" class="btn">Close</button>
            </div>
        </div>
    </div>
    
    <script>
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            row.style.display = row.textContent.toLowerCase().includes(search) ? '' : 'none';
        });
    });
    
    function viewPayment(payment) {
        const html = `
            <div style="line-height:1.8">
                <p><strong>M-Pesa Transaction ID:</strong> ${payment.mpesa_tx_id}</p>
                <p><strong>Meter ID:</strong> ${payment.meter_id || 'N/A'}</p>
                <p><strong>Customer:</strong> ${payment.customer_name || 'N/A'}</p>
                <p><strong>Phone:</strong> ${payment.phone}</p>
                <p><strong>Amount:</strong> KES ${parseFloat(payment.amount).toFixed(2)}</p>
                <p><strong>Paybill:</strong> ${payment.paybill}</p>
                <p><strong>Account:</strong> ${payment.account_number}</p>
                <p><strong>Status:</strong> <span class="badge">${payment.status}</span></p>
                <p><strong>Date:</strong> ${payment.tx_date}</p>
                <p><strong>Token:</strong> ${payment.token_value || 'Not generated yet'}</p>
                ${payment.error_msg ? '<p><strong>Error:</strong> <span style="color:red">' + payment.error_msg + '</span></p>' : ''}
                ${payment.processed_at ? '<p><strong>Processed:</strong> ' + payment.processed_at + '</p>' : ''}
            </div>
        `;
        document.getElementById('paymentDetails').innerHTML = html;
        document.getElementById('viewModal').classList.add('active');
    }
    
    function closeModal() {
        document.getElementById('viewModal').classList.remove('active');
    }
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>