<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['client_user_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$clientId = $_SESSION['client_id'];
$success = $error = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        try {
            $customerId = 'CUST-' . strtoupper(substr(md5(uniqid()), 0, 8));
            $stmt = $db->prepare("
                INSERT INTO customers 
                (client_id, customer_id, first_name, last_name, phone, email, id_number, address)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $clientId,
                $customerId,
                $_POST['first_name'],
                $_POST['last_name'],
                $_POST['phone'],
                $_POST['email'],
                $_POST['id_number'],
                $_POST['address']
            ]);
            $success = 'Customer created successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'toggle') {
        $stmt = $db->prepare("UPDATE customers SET is_active = NOT is_active WHERE id = ? AND client_id = ?");
        $stmt->execute([$_POST['id'], $clientId]);
        $success = 'Customer status updated';
    }
}

// Get customers
$customers = $db->prepare("
    SELECT c.*, 
           m.meter_id,
           COUNT(DISTINCT t.id) as token_count,
           COALESCE(SUM(t.amount), 0) as total_spent
    FROM customers c
    LEFT JOIN meter_assignments ma ON c.id = ma.customer_id AND ma.is_active = 1
    LEFT JOIN meters m ON ma.meter_id = m.id
    LEFT JOIN tokens t ON c.id = t.customer_id
    WHERE c.client_id = ?
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$customers->execute([$clientId]);
$customers = $customers->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Customer Management</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openModal()" class="btn btn-orange">
                        <i class="fas fa-plus"></i> Add Customer
                    </button>
                    <span><?= htmlspecialchars($_SESSION['client_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>All Customers (<?= count($customers) ?>)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Customer ID</th>
                                <th>Name</th>
                                <th>Phone</th>
                                <th>Email</th>
                                <th>Meter</th>
                                <th>Tokens</th>
                                <th>Total Spent</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($customers as $c): ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($c['customer_id']) ?></strong></td>
                                    <td><?= htmlspecialchars($c['first_name'].' '.$c['last_name']) ?></td>
                                    <td><?= htmlspecialchars($c['phone']) ?></td>
                                    <td><?= htmlspecialchars($c['email'] ?: 'N/A') ?></td>
                                    <td>
                                        <?php if ($c['meter_id']): ?>
                                            <code><?= htmlspecialchars($c['meter_id']) ?></code>
                                        <?php else: ?>
                                            <span class="badge badge-warning">No meter</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= $c['token_count'] ?></td>
                                    <td>KES <?= number_format($c['total_spent'], 2) ?></td>
                                    <td>
                                        <?php if ($c['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge badge-danger">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <form method="POST" style="display:inline">
                                            <input type="hidden" name="action" value="toggle">
                                            <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                            <button type="submit" class="btn btn-sm" onclick="return confirm('Toggle status?')">
                                                <i class="fas fa-power-off"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Customer Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New Customer</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>First Name *</label>
                            <input type="text" name="first_name" required>
                        </div>
                        <div class="form-group">
                            <label>Last Name *</label>
                            <input type="text" name="last_name" required>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Phone *</label>
                            <input type="text" name="phone" placeholder="0745600377" required>
                            <small style="color:#666">Will be normalized to +254...</small>
                        </div>
                        <div class="form-group">
                            <label>Email</label>
                            <input type="email" name="email">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>ID Number</label>
                        <input type="text" name="id_number">
                    </div>
                    
                    <div class="form-group">
                        <label>Address</label>
                        <textarea name="address" rows="2"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Create Customer</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openModal() {
        document.getElementById('addModal').classList.add('active');
    }
    function closeModal() {
        document.getElementById('addModal').classList.remove('active');
    }
    
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>