<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';
require_once '../../includes/SMSService.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$success = $error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'delete') {
        try {
            $stmt = $db->prepare("DELETE FROM tokens WHERE id = ?");
            $stmt->execute([$_POST['id']]);
            $success = 'Token deleted successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'resend_sms') {
        $tokenId = $_POST['token_id'];
        $phone = $_POST['phone'];
        
        $stmt = $db->prepare("
            SELECT t.*, m.meter_id 
            FROM tokens t 
            JOIN meters m ON t.meter_id = m.id 
            WHERE t.id = ?
        ");
        $stmt->execute([$tokenId]);
        $token = $stmt->fetch();
        
        if ($token) {
            $sms = new SMSService();
            list($smsOk, $smsResp) = $sms->sendToken(
                $phone,
                $token['meter_id'],
                $token['token_value'],
                $token['amount'],
                $token['units'] ?? 0
            );
            
            if ($smsOk) {
                $stmt = $db->prepare("
                    UPDATE tokens 
                    SET status = 'DELIVERED', delivered_at = NOW(), sms_to = ?, sms_response = ?
                    WHERE id = ?
                ");
                $stmt->execute([$phone, $smsResp, $tokenId]);
                $success = 'SMS sent successfully to ' . $phone;
            } else {
                $error = 'Failed to send SMS';
            }
        }
    }
}

// Filters
$filterClient = $_GET['client'] ?? '';
$filterType = $_GET['type'] ?? '';
$filterStatus = $_GET['status'] ?? '';
$filterDate = $_GET['date'] ?? '';

$query = "
    SELECT t.*, 
           c.company_name, c.client_code,
           m.meter_id,
           cu.customer_id, CONCAT(cu.first_name, ' ', cu.last_name) as customer_name, cu.phone,
           p.mpesa_tx_id
    FROM tokens t
    JOIN clients c ON t.client_id = c.id
    JOIN meters m ON t.meter_id = m.id
    JOIN customers cu ON t.customer_id = cu.id
    LEFT JOIN payments p ON t.payment_id = p.id
    WHERE 1=1
";

$params = [];
if ($filterClient) {
    $query .= " AND c.id = ?";
    $params[] = $filterClient;
}
if ($filterType) {
    $query .= " AND t.token_type = ?";
    $params[] = $filterType;
}
if ($filterStatus) {
    $query .= " AND t.status = ?";
    $params[] = $filterStatus;
}
if ($filterDate) {
    $query .= " AND DATE(t.created_at) = ?";
    $params[] = $filterDate;
}

$query .= " ORDER BY t.created_at DESC LIMIT 100";

$stmt = $db->prepare($query);
$stmt->execute($params);
$tokens = $stmt->fetchAll();

// Get clients for filter
$clients = $db->query("SELECT id, company_name FROM clients ORDER BY company_name")->fetchAll();

// Statistics
$stats = [
    'total' => $db->query("SELECT COUNT(*) as c FROM tokens")->fetch()['c'],
    'today' => $db->query("SELECT COUNT(*) as c FROM tokens WHERE DATE(created_at)=CURDATE()")->fetch()['c'],
    'delivered' => $db->query("SELECT COUNT(*) as c FROM tokens WHERE status='DELIVERED'")->fetch()['c'],
    'failed' => $db->query("SELECT COUNT(*) as c FROM tokens WHERE status='FAILED'")->fetch()['c'],
    'manual' => $db->query("SELECT COUNT(*) as c FROM tokens WHERE is_manual=1")->fetch()['c']
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Token Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Token Management</h1>
                <div class="flex items-center gap-4">
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <!-- Statistics -->
                <div class="grid grid-cols-3 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Total Tokens</h3>
                        <div class="value"><?= $stats['total'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Today</h3>
                        <div class="value"><?= $stats['today'] ?></div>
                    </div>
                    <div class="stat-card">
                        <h3>Delivered</h3>
                        <div class="value"><?= $stats['delivered'] ?></div>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4 mb-4">
                    <div class="stat-card" style="border-left-color:#dc3545">
                        <h3>Failed</h3>
                        <div class="value"><?= $stats['failed'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Manual</h3>
                        <div class="value"><?= $stats['manual'] ?></div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="bg-white p-4 rounded shadow mb-4">
                    <form method="GET" class="grid grid-cols-5 gap-4">
                        <div class="form-group">
                            <label>Client</label>
                            <select name="client">
                                <option value="">All Clients</option>
                                <?php foreach ($clients as $c): ?>
                                    <option value="<?= $c['id'] ?>" <?= $filterClient == $c['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($c['company_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Type</label>
                            <select name="type">
                                <option value="">All Types</option>
                                <option value="VENDING" <?= $filterType == 'VENDING' ? 'selected' : '' ?>>Vending</option>
                                <option value="CLEAR_CREDIT" <?= $filterType == 'CLEAR_CREDIT' ? 'selected' : '' ?>>Clear Credit</option>
                                <option value="CLEAR_TAMPER" <?= $filterType == 'CLEAR_TAMPER' ? 'selected' : '' ?>>Clear Tamper</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Status</label>
                            <select name="status">
                                <option value="">All Status</option>
                                <option value="DELIVERED" <?= $filterStatus == 'DELIVERED' ? 'selected' : '' ?>>Delivered</option>
                                <option value="CREATED" <?= $filterStatus == 'CREATED' ? 'selected' : '' ?>>Created</option>
                                <option value="FAILED" <?= $filterStatus == 'FAILED' ? 'selected' : '' ?>>Failed</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Date</label>
                            <input type="date" name="date" value="<?= htmlspecialchars($filterDate) ?>">
                        </div>
                        <div class="form-group" style="display:flex;align-items:flex-end;gap:0.5rem">
                            <button type="submit" class="btn btn-primary">Filter</button>
                            <a href="tokens.php" class="btn">Reset</a>
                        </div>
                    </form>
                </div>
                
                <!-- Tokens Table -->
                <div class="table-container">
                    <div class="table-header">
                        <h2>Tokens (Last 100)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Client</th>
                                <th>Meter</th>
                                <th>Customer</th>
                                <th>Amount</th>
                                <th>Units</th>
                                <th>Token</th>
                                <th>Source</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tokens as $t): ?>
                                <tr>
                                    <td><?= date('Y-m-d H:i', strtotime($t['created_at'])) ?></td>
                                    <td>
                                        <?php
                                        $typeColors = [
                                            'VENDING' => 'badge-success',
                                            'CLEAR_CREDIT' => 'badge-info',
                                            'CLEAR_TAMPER' => 'badge-warning'
                                        ];
                                        ?>
                                        <span class="badge <?= $typeColors[$t['token_type']] ?? 'badge-info' ?>">
                                            <?= str_replace('_', ' ', $t['token_type']) ?>
                                        </span>
                                    </td>
                                    <td><?= htmlspecialchars($t['company_name']) ?></td>
                                    <td><code><?= htmlspecialchars($t['meter_id']) ?></code></td>
                                    <td><?= htmlspecialchars($t['customer_name']) ?></td>
                                    <td>KES <?= number_format($t['amount'], 2) ?></td>
                                    <td><?= number_format($t['units'], 2) ?></td>
                                    <td><code style="font-size:0.7rem"><?= htmlspecialchars($t['token_value']) ?></code></td>
                                    <td>
                                        <?php if ($t['is_manual']): ?>
                                            <span class="badge badge-info">Manual</span>
                                        <?php else: ?>
                                            <span class="badge badge-success">Auto</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php
                                        $statusColors = [
                                            'DELIVERED' => 'badge-success',
                                            'CREATED' => 'badge-warning',
                                            'FAILED' => 'badge-danger'
                                        ];
                                        ?>
                                        <span class="badge <?= $statusColors[$t['status']] ?? 'badge-info' ?>">
                                            <?= $t['status'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <button onclick="viewToken(<?= htmlspecialchars(json_encode($t)) ?>)" class="btn btn-sm btn-primary">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button onclick="resendSMS(<?= $t['id'] ?>, '<?= htmlspecialchars($t['phone']) ?>')" class="btn btn-sm" style="background:#f78510;color:white">
                                            <i class="fas fa-sms"></i>
                                        </button>
                                        <form method="POST" style="display:inline" onsubmit="return confirm('Delete this token?')">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $t['id'] ?>">
                                            <button type="submit" class="btn btn-sm" style="background:#dc3545;color:white">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- View Token Modal -->
    <div id="viewModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Token Details</h3>
                <button onclick="closeViewModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <div class="modal-body" id="tokenDetails">
                <!-- Populated by JS -->
            </div>
            <div class="modal-footer">
                <button onclick="closeViewModal()" class="btn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Resend SMS Modal -->
    <div id="smsModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Resend Token SMS</h3>
                <button onclick="closeSMSModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="resend_sms">
                <input type="hidden" name="token_id" id="smsTokenId">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Send to Phone Number *</label>
                        <input type="text" name="phone" id="smsPhone" placeholder="+254745600377" required>
                        <small style="color:#666">Will be normalized to +254 format</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeSMSModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Send SMS</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    
    function viewToken(token) {
        const html = `
            <div style="line-height:1.8">
                <p><strong>Token Value:</strong> <code>${token.token_value}</code></p>
                <p><strong>Type:</strong> ${token.token_type.replace('_', ' ')}</p>
                <p><strong>Client:</strong> ${token.company_name}</p>
                <p><strong>Meter ID:</strong> ${token.meter_id}</p>
                <p><strong>Customer:</strong> ${token.customer_name} (${token.customer_id})</p>
                <p><strong>Phone:</strong> ${token.phone}</p>
                <p><strong>Amount:</strong> KES ${parseFloat(token.amount).toFixed(2)}</p>
                <p><strong>Units:</strong> ${parseFloat(token.units || 0).toFixed(2)}</p>
                <p><strong>Vend By Unit:</strong> ${token.is_vend_by_unit ? 'Yes' : 'No'}</p>
                <p><strong>Source:</strong> ${token.is_manual ? 'Manual' : 'Automatic'}</p>
                <p><strong>Status:</strong> <span class="badge">${token.status}</span></p>
                <p><strong>Created:</strong> ${token.created_at}</p>
                ${token.delivered_at ? '<p><strong>Delivered:</strong> ' + token.delivered_at + '</p>' : ''}
                ${token.sms_to ? '<p><strong>SMS Sent To:</strong> ' + token.sms_to + '</p>' : ''}
                ${token.mpesa_tx_id ? '<p><strong>M-Pesa TX:</strong> ' + token.mpesa_tx_id + '</p>' : ''}
            </div>
        `;
        document.getElementById('tokenDetails').innerHTML = html;
        document.getElementById('viewModal').classList.add('active');
    }
    
    function closeViewModal() {
        document.getElementById('viewModal').classList.remove('active');
    }
    
    function resendSMS(tokenId, phone) {
        document.getElementById('smsTokenId').value = tokenId;
        document.getElementById('smsPhone').value = phone;
        document.getElementById('smsModal').classList.add('active');
    }
    
    function closeSMSModal() {
        document.getElementById('smsModal').classList.remove('active');
    }
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>