<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();

// Date range filter
$startDate = $_GET['start_date'] ?? date('Y-m-01');
$endDate = $_GET['end_date'] ?? date('Y-m-d');

// Revenue by client
$revenueByClient = $db->prepare("
    SELECT c.company_name, 
           COUNT(p.id) as tx_count,
           SUM(p.amount) as total_revenue
    FROM clients c
    LEFT JOIN payments p ON c.id = p.client_id 
        AND p.status = 'PROCESSED'
        AND DATE(p.tx_date) BETWEEN ? AND ?
    GROUP BY c.id
    ORDER BY total_revenue DESC
");
$revenueByClient->execute([$startDate, $endDate]);
$revenueByClient = $revenueByClient->fetchAll();

// Daily revenue trend
$dailyRevenue = $db->prepare("
    SELECT DATE(tx_date) as date,
           COUNT(id) as tx_count,
           SUM(amount) as revenue
    FROM payments
    WHERE status = 'PROCESSED'
    AND DATE(tx_date) BETWEEN ? AND ?
    GROUP BY DATE(tx_date)
    ORDER BY date ASC
");
$dailyRevenue->execute([$startDate, $endDate]);
$dailyRevenue = $dailyRevenue->fetchAll();

// Top customers
$topCustomers = $db->prepare("
    SELECT cu.customer_id,
           CONCAT(cu.first_name, ' ', cu.last_name) as name,
           c.company_name,
           COUNT(t.id) as token_count,
           SUM(t.amount) as total_spent
    FROM customers cu
    JOIN clients c ON cu.client_id = c.id
    LEFT JOIN tokens t ON cu.id = t.customer_id
        AND DATE(t.created_at) BETWEEN ? AND ?
    GROUP BY cu.id
    ORDER BY total_spent DESC
    LIMIT 10
");
$topCustomers->execute([$startDate, $endDate]);
$topCustomers = $topCustomers->fetchAll();

// Token distribution
$tokenStats = $db->prepare("
    SELECT token_type,
           COUNT(*) as count,
           SUM(CASE WHEN status='DELIVERED' THEN 1 ELSE 0 END) as delivered,
           SUM(CASE WHEN status='FAILED' THEN 1 ELSE 0 END) as failed
    FROM tokens
    WHERE DATE(created_at) BETWEEN ? AND ?
    GROUP BY token_type
");
$tokenStats->execute([$startDate, $endDate]);
$tokenStats = $tokenStats->fetchAll();

// Overall stats
$stats = [
    'total_revenue' => $db->prepare("SELECT COALESCE(SUM(amount),0) as c FROM payments WHERE status='PROCESSED' AND DATE(tx_date) BETWEEN ? AND ?"),
    'total_tx' => $db->prepare("SELECT COUNT(*) as c FROM payments WHERE DATE(tx_date) BETWEEN ? AND ?"),
    'total_tokens' => $db->prepare("SELECT COUNT(*) as c FROM tokens WHERE DATE(created_at) BETWEEN ? AND ?"),
    'success_rate' => $db->prepare("SELECT (COUNT(CASE WHEN status='PROCESSED' THEN 1 END) * 100.0 / NULLIF(COUNT(*), 0)) as c FROM payments WHERE DATE(tx_date) BETWEEN ? AND ?")
];

foreach ($stats as $key => $stmt) {
    $stmt->execute([$startDate, $endDate]);
    $stats[$key] = $stmt->fetch()['c'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics Dashboard</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Analytics Dashboard</h1>
                <div class="flex items-center gap-4">
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <!-- Date Range Filter -->
                <div class="bg-white p-4 rounded shadow mb-4">
                    <form method="GET" class="flex gap-4 items-end">
                        <div class="form-group">
                            <label>Start Date</label>
                            <input type="date" name="start_date" value="<?= htmlspecialchars($startDate) ?>" required>
                        </div>
                        <div class="form-group">
                            <label>End Date</label>
                            <input type="date" name="end_date" value="<?= htmlspecialchars($endDate) ?>" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Apply</button>
                        <a href="analytics.php" class="btn">Reset</a>
                    </form>
                </div>
                
                <!-- Overview Stats -->
                <div class="grid grid-cols-4 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Total Revenue</h3>
                        <div class="value">KES <?= number_format($stats['total_revenue'], 2) ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Transactions</h3>
                        <div class="value"><?= $stats['total_tx'] ?></div>
                    </div>
                    <div class="stat-card">
                        <h3>Tokens Issued</h3>
                        <div class="value"><?= $stats['total_tokens'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Success Rate</h3>
                        <div class="value"><?= number_format($stats['success_rate'] ?? 0, 1) ?>%</div>
                    </div>
                </div>
                
                <!-- Charts -->
                <div class="grid grid-cols-2 gap-4 mb-4">
                    <div class="bg-white p-4 rounded shadow">
                        <h3 class="text-blue mb-4">Daily Revenue Trend</h3>
                        <canvas id="revenueChart" style="max-height:300px"></canvas>
                    </div>
                    <div class="bg-white p-4 rounded shadow">
                        <h3 class="text-blue mb-4">Revenue by Client</h3>
                        <canvas id="clientChart" style="max-height:300px"></canvas>
                    </div>
                </div>
                
                <!-- Top Customers Table -->
                <div class="table-container mb-4">
                    <div class="table-header">
                        <h2>Top 10 Customers</h2>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>Customer</th>
                                <th>Client</th>
                                <th>Tokens</th>
                                <th>Total Spent</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $rank = 1; foreach ($topCustomers as $c): ?>
                                <tr>
                                    <td><strong>#<?= $rank++ ?></strong></td>
                                    <td>
                                        <?= htmlspecialchars($c['name']) ?><br>
                                        <small><?= htmlspecialchars($c['customer_id']) ?></small>
                                    </td>
                                    <td><?= htmlspecialchars($c['company_name']) ?></td>
                                    <td><?= $c['token_count'] ?></td>
                                    <td><strong>KES <?= number_format($c['total_spent'], 2) ?></strong></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Token Statistics -->
                <div class="table-container">
                    <div class="table-header">
                        <h2>Token Distribution</h2>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Token Type</th>
                                <th>Total</th>
                                <th>Delivered</th>
                                <th>Failed</th>
                                <th>Success Rate</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tokenStats as $ts): ?>
                                <tr>
                                    <td><strong><?= str_replace('_', ' ', $ts['token_type']) ?></strong></td>
                                    <td><?= $ts['count'] ?></td>
                                    <td><span class="badge badge-success"><?= $ts['delivered'] ?></span></td>
                                    <td><span class="badge badge-danger"><?= $ts['failed'] ?></span></td>
                                    <td><?= number_format(($ts['delivered'] / max($ts['count'], 1)) * 100, 1) ?>%</td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Revenue Trend Chart
    const revenueCtx = document.getElementById('revenueChart').getContext('2d');
    new Chart(revenueCtx, {
        type: 'line',
        data: {
            labels: <?= json_encode(array_column($dailyRevenue, 'date')) ?>,
            datasets: [{
                label: 'Revenue (KES)',
                data: <?= json_encode(array_column($dailyRevenue, 'revenue')) ?>,
                borderColor: '#1b405f',
                backgroundColor: 'rgba(27, 64, 95, 0.1)',
                tension: 0.4,
                fill: true
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { display: false }
            },
            scales: {
                y: { beginAtZero: true }
            }
        }
    });
    
    // Client Revenue Chart
    const clientCtx = document.getElementById('clientChart').getContext('2d');
    new Chart(clientCtx, {
        type: 'bar',
        data: {
            labels: <?= json_encode(array_column($revenueByClient, 'company_name')) ?>,
            datasets: [{
                label: 'Revenue (KES)',
                data: <?= json_encode(array_column($revenueByClient, 'total_revenue')) ?>,
                backgroundColor: '#f78510',
                borderColor: '#f78510',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { display: false }
            },
            scales: {
                y: { beginAtZero: true }
            }
        }
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>